/**
 * Logic for Snippets Page
 */

let monacoEditor = null;
let snippetsMap = {};
let allSnippets = [];

$(document).ready(function() {
    Utils.initTheme();
    loadSnippets();

    // Filter Events
    $('#filterLang, #filterSearch').on('input change', function() {
        applyFilters();
    });

    // Initialize Monaco
    require.config({ paths: { 'vs': 'plugins/monaco-editor/min/vs' } });
    require(['vs/editor/editor.main'], function() {
        monacoEditor = monaco.editor.create(document.getElementById('monaco-container'), {
            value: '',
            language: 'plaintext',
            theme: $('html').attr('data-bs-theme') === 'dark' ? 'vs-dark' : 'vs',
            automaticLayout: true,
            minimap: { enabled: false },
            fontSize: 14,
            fontFamily: 'Monaco, Consolas, "Courier New", monospace',
            scrollBeyondLastLine: false
        });

        // Raccourci Ctrl+Shift+Haut : Déplacer la ligne vers le haut
        monacoEditor.addCommand(monaco.KeyMod.CtrlCmd | monaco.KeyMod.Shift | monaco.KeyCode.UpArrow, function() {
            monacoEditor.trigger('keyboard', 'editor.action.moveLinesUpAction');
        });

        // Raccourci Ctrl+Shift+Bas : Déplacer la ligne vers le bas
        monacoEditor.addCommand(monaco.KeyMod.CtrlCmd | monaco.KeyMod.Shift | monaco.KeyCode.DownArrow, function() {
            monacoEditor.trigger('keyboard', 'editor.action.moveLinesDownAction');
        });
    });

    $('#snippetLang').on('change', function() {
        if (monacoEditor) {
            const lang = Utils.getMonacoLanguage($(this).val());
            monaco.editor.setModelLanguage(monacoEditor.getModel(), lang);
        }
    });

    $('#editorModal').on('shown.bs.modal', function () {
        if (monacoEditor) {
            monacoEditor.layout();
        }
    });

    $('#sublimeModal').on('shown.bs.modal', function () {
        $('#sublimeInput').val('').focus();
    });
});

function loadSnippets() {
    allSnippets = DB.getSnippets();
    initFilters();
    applyFilters();
}

function initFilters() {
    // Populate Language Dropdown
    const languages = [...new Set(allSnippets.map(s => s.language))].sort();
    const $select = $('#filterLang');
    const currentVal = $select.val();
    
    $select.html('<option value="">Tous les langages</option>');
    languages.forEach(lang => {
        $select.append(`<option value="${lang}">${lang}</option>`);
    });
    if (currentVal) $select.val(currentVal);

    // Populate Autocomplete
    const $datalist = $('#searchOptions');
    $datalist.empty();
    allSnippets.forEach(s => {
        if (s.label) $datalist.append(`<option value="${s.label}">`);
        if (s.description) $datalist.append(`<option value="${s.description}">`);
    });
}

function applyFilters() {
    const langFilter = $('#filterLang').val();
    const searchFilter = $('#filterSearch').val().toLowerCase();

    const filtered = allSnippets.filter(s => {
        const matchLang = !langFilter || s.language === langFilter;
        const matchSearch = !searchFilter || 
            (s.label && s.label.toLowerCase().includes(searchFilter)) ||
            (s.description && s.description.toLowerCase().includes(searchFilter)) ||
            (s.code && s.code.toLowerCase().includes(searchFilter));
        
        return matchLang && matchSearch;
    });

    renderSnippets(filtered);
}

function renderSnippets(snippets) {
    const container = $('#snippetsList');
    container.empty();
    snippetsMap = {};

    if (snippets.length === 0) {
        container.html('<div class="text-center text-muted py-5">Aucun snippet configuré.</div>');
        return;
    }

    // Group by language
    const grouped = snippets.reduce((acc, s) => {
        acc[s.language] = acc[s.language] || [];
        acc[s.language].push(s);
        return acc;
    }, {});

    Object.keys(grouped).sort().forEach(lang => {
        const langHeader = `
            <div class="lang-header">
                <h5 class="mb-0 text-primary text-uppercase fw-bold ps-2 border-start border-4 border-primary">
                    ${lang}
                </h5>
            </div>
            <div class="row g-3 mb-4">
        `;
        container.append(langHeader);

        const row = $('<div class="row g-3 mb-4"></div>');
        grouped[lang].forEach(s => {
            snippetsMap[s.id] = s;
            const card = `
                <div class="col-md-6 col-lg-4">
                    <div class="snippet-card h-100 p-3 position-relative">
                        <div class="d-flex justify-content-between align-items-start mb-2">
                            <span class="badge bg-light text-dark border font-monospace">${Utils.escapeHtml(s.label)}</span>
                            <button class="btn btn-sm btn-link text-secondary p-0" onclick="editSnippet(${s.id})">
                                <i class="bi bi-pencil-square"></i>
                            </button>
                        </div>
                        <p class="small text-muted mb-2">${Utils.escapeHtml(s.description || 'Sans description')}</p>
                        <div class="snippet-code rounded p-2 font-monospace small text-truncate">
                            ${Utils.escapeHtml(s.code)}
                        </div>
                    </div>
                </div>
            `;
            row.append(card);
        });
        container.append(row);
    });
}

function openModal() {
    $('#snippetForm')[0].reset();
    $('#snippetId').val(0);
    $('#btnDelete').addClass('d-none');
    $('#modalTitle').text('Nouveau Snippet');
    
    if (monacoEditor) {
        monacoEditor.setValue('');
        monaco.editor.setModelLanguage(monacoEditor.getModel(), 'plaintext');
    }
    
    $('#editorModal').modal('show');
}

function editSnippet(id) {
    const snippet = snippetsMap[id];
    if (!snippet) return;

    $('#snippetId').val(snippet.id);
    $('#snippetLang').val(snippet.language);
    $('#snippetLabel').val(snippet.label);
    $('#snippetDesc').val(snippet.description);
    $('#modalTitle').text('Modifier Snippet');
    
    if (monacoEditor) {
        monacoEditor.setValue(snippet.code);
        const lang = Utils.getMonacoLanguage(snippet.language);
        monaco.editor.setModelLanguage(monacoEditor.getModel(), lang);
    }
    
    $('#btnDelete').removeClass('d-none');
    $('#editorModal').modal('show');
}

function saveSnippet() {
    const form = $('#snippetForm');
    if (!form[0].checkValidity()) {
        form[0].reportValidity();
        return;
    }

    const content = monacoEditor ? monacoEditor.getValue() : '';
    
    const data = {
        id: $('#snippetId').val(),
        language: $('#snippetLang').val(),
        label: $('#snippetLabel').val(),
        description: $('#snippetDesc').val(),
        code: content
    };
    
    DB.saveSnippet(data);
    $('#editorModal').modal('hide');
    Utils.showToast('Snippet enregistré !');
    loadSnippets();
}

function deleteSnippet() {
    const id = $('#snippetId').val();
    if (id && id != 0) {
        if (confirm('Supprimer ce snippet ?')) {
            DB.deleteSnippet(id);
            $('#editorModal').modal('hide');
            loadSnippets();
            Utils.showToast('Snippet supprimé');
        }
    }
}

// Sublime Text Converter
function convertSublime() {
    const xmlContent = $('#sublimeInput').val();
    if (!xmlContent.trim()) {
        Utils.showToast('Veuillez coller un snippet XML', 'warning');
        return;
    }

    try {
        const parser = new DOMParser();
        const xmlDoc = parser.parseFromString(xmlContent, "text/xml");
        
        // Check for parse errors
        if (xmlDoc.getElementsByTagName("parsererror").length > 0) {
            throw new Error("Erreur de parsing XML");
        }

        const content = xmlDoc.querySelector('content') ? xmlDoc.querySelector('content').textContent : '';
        const tabTrigger = xmlDoc.querySelector('tabTrigger') ? xmlDoc.querySelector('tabTrigger').textContent : '';
        const description = xmlDoc.querySelector('description') ? xmlDoc.querySelector('description').textContent : '';
        const scope = xmlDoc.querySelector('scope') ? xmlDoc.querySelector('scope').textContent : '';

        // Map Scope to Language
        let language = 'text';
        if (scope.includes('js') || scope.includes('javascript')) language = 'javascript';
        else if (scope.includes('php')) language = 'php';
        else if (scope.includes('html')) language = 'html';
        else if (scope.includes('css')) language = 'css';
        else if (scope.includes('python')) language = 'python';
        else if (scope.includes('sql')) language = 'sql';
        else if (scope.includes('shell') || scope.includes('bash')) language = 'shell';
        else if (scope.includes('markdown')) language = 'markdown';
        else if (scope.includes('perl')) language = 'perl';
        else if (scope.includes('json')) language = 'json';

        // Close Converter Modal
        const sublimeModalEl = document.getElementById('sublimeModal');
        const sublimeModal = bootstrap.Modal.getInstance(sublimeModalEl);
        sublimeModal.hide();

        // Open Editor Modal and Fill
        openModal();
        
        // Wait a bit for modal to init (optional but safer)
        setTimeout(() => {
            $('#snippetLang').val(language).trigger('change');
            $('#snippetLabel').val(tabTrigger);
            $('#snippetDesc').val(description);
            
            if (monacoEditor) {
                monacoEditor.setValue(content);
                // Trigger language change logic
                const langMode = Utils.getMonacoLanguage(language);
                monaco.editor.setModelLanguage(monacoEditor.getModel(), langMode);
            }
            
            Utils.showToast('Snippet converti avec succès');
        }, 200);

    } catch (e) {
        console.error(e);
        Utils.showToast('Format XML invalide', 'danger');
    }
}
