/**
 * Plugins pour l'éditeur Copie (Local)
 */
const CopiePlugins = {
    editor: null,

    /**
     * Initialise les plugins avec l'instance de l'éditeur Monaco
     * @param {monaco.editor.IStandaloneCodeEditor} editorInstance 
     */
    init: function(editorInstance) {
        this.editor = editorInstance;
    },

    /**
     * Formate le JSON (Pretty Print)
     */
    formatJSON: function() {
        if (!this.editor) return;
        try {
            const value = this.editor.getValue();
            if (!value.trim()) return;
            
            const json = JSON.parse(value);
            const formatted = JSON.stringify(json, null, 4);
            
            this.editor.executeEdits('formatJSON', [{
                range: this.editor.getModel().getFullModelRange(),
                text: formatted
            }]);
            
            this.editor.getAction('editor.action.formatDocument').run();
        } catch (e) {
            this.showError('JSON Invalide: ' + e.message);
        }
    },

    /**
     * Minifie le JSON
     */
    minifyJSON: function() {
        if (!this.editor) return;
        try {
            const value = this.editor.getValue();
            if (!value.trim()) return;

            const json = JSON.parse(value);
            const minified = JSON.stringify(json);
            
            this.editor.executeEdits('minifyJSON', [{
                range: this.editor.getModel().getFullModelRange(),
                text: minified
            }]);
        } catch (e) {
            this.showError('JSON Invalide: ' + e.message);
        }
    },

    /**
     * Trie les lignes du texte
     * @param {string} direction 'asc' ou 'desc'
     */
    sortLines: function(direction = 'asc') {
        if (!this.editor) return;
        const model = this.editor.getModel();
        const selection = this.editor.getSelection();
        let range, text;

        if (!selection || selection.isEmpty()) {
            range = model.getFullModelRange();
            text = model.getValue();
        } else {
            // Expand selection to full lines
            range = new window.monaco.Range(
                selection.startLineNumber,
                1,
                selection.endLineNumber,
                model.getLineMaxColumn(selection.endLineNumber)
            );
            text = model.getValueInRange(range);
        }

        const lines = text.split(/\r?\n/);
        
        if (direction === 'asc') {
            lines.sort((a, b) => a.localeCompare(b));
        } else {
            lines.sort((a, b) => b.localeCompare(a));
        }
        
        const newText = lines.join('\n');
        this.editor.executeEdits('sortLines', [{
            range: range,
            text: newText
        }]);
    },

    /**
     * Supprime les lignes en double
     */
    removeDuplicates: function() {
        if (!this.editor) return;
        const model = this.editor.getModel();
        const selection = this.editor.getSelection();
        let range, text;

        if (!selection || selection.isEmpty()) {
            range = model.getFullModelRange();
            text = model.getValue();
        } else {
            range = new window.monaco.Range(
                selection.startLineNumber,
                1,
                selection.endLineNumber,
                model.getLineMaxColumn(selection.endLineNumber)
            );
            text = model.getValueInRange(range);
        }

        const lines = text.split(/\r?\n/);
        const uniqueLines = [...new Set(lines)];
        
        const newText = uniqueLines.join('\n');
        this.editor.executeEdits('removeDuplicates', [{
            range: range,
            text: newText
        }]);
    },

    /**
     * Récupère les symboles (fonctions, classes, etc.) du fichier courant
     * @returns {Array} Liste des symboles {line, text, type}
     */
    getSymbols: function() {
        if (!this.editor) return [];
        const model = this.editor.getModel();
        const lang = model.getLanguageId();
        const text = model.getValue();
        const lines = text.split(/\r?\n/);
        const symbols = [];

        // Regex definitions per language
        const regexes = {
            'javascript': [
                { re: /function\s+([a-zA-Z0-9_$]+)/, type: 'Function' },
                { re: /const\s+([a-zA-Z0-9_$]+)\s*=\s*(\(.*\)|async\s*\(.*\)|.*)\s*=>/, type: 'Arrow Func' },
                { re: /class\s+([a-zA-Z0-9_$]+)/, type: 'Class' }
            ],
            'php': [
                { re: /function\s+([a-zA-Z0-9_]+)/, type: 'Function' },
                { re: /class\s+([a-zA-Z0-9_]+)/, type: 'Class' },
                { re: /trait\s+([a-zA-Z0-9_]+)/, type: 'Trait' }
            ],
            'python': [
                { re: /^def\s+([a-zA-Z0-9_]+)/, type: 'Def' },
                { re: /^class\s+([a-zA-Z0-9_]+)/, type: 'Class' }
            ],
            'perl': [
                { re: /sub\s+([a-zA-Z0-9_]+)/, type: 'Sub' },
                { re: /package\s+([a-zA-Z0-9_:]+)/, type: 'Package' }
            ],
            'shell': [
                { re: /^([a-zA-Z0-9_-]+)\(\)/, type: 'Function' },
                { re: /function\s+([a-zA-Z0-9_-]+)/, type: 'Function' }
            ],
            'markdown': [
                { re: /^(#{1,6})\s+(.*)/, type: 'Header' }
            ],
            'sql': [
                { re: /CREATE\s+(TABLE|VIEW|PROCEDURE|FUNCTION)\s+([a-zA-Z0-9_]+)/i, type: 'Definition' }
            ]
        };

        const currentRegexes = regexes[lang] || [];

        lines.forEach((line, index) => {
            const lineNumber = index + 1;
            
            for (const rule of currentRegexes) {
                const match = line.match(rule.re);
                if (match) {
                    // For SQL, the name is in group 2, for others usually group 1
                    // For Markdown, group 2 is the text
                    let name = match[1];
                    if (lang === 'sql') name = `${match[1]} ${match[2]}`;
                    if (lang === 'markdown') name = `${match[1]} ${match[2]}`;

                    symbols.push({
                        line: lineNumber,
                        text: name,
                        type: rule.type,
                        fullLine: line.trim()
                    });
                    break; // Only one symbol per line usually
                }
            }
        });

        return symbols;
    },

    /**
     * Affiche une erreur
     */
    showError: function(msg) {
        if (typeof Utils !== 'undefined' && Utils.showToast) {
            Utils.showToast(msg, 'danger');
        } else {
            alert(msg);
        }
    },

    /**
     * Met à jour la visibilité des outils en fonction du langage
     */
    updateVisibility: function(language) {
        const jsonTools = document.getElementById('jsonToolsDropdown');
        if (jsonTools) {
            if (language === 'json') {
                jsonTools.classList.remove('d-none');
            } else {
                jsonTools.classList.add('d-none');
            }
        }
    }
};
