/**
 * Logic for Index Page (Copies Manager)
 */

let currentFilter = 'all';
let currentSort = localStorage.getItem('copie_sort') || 'date_desc';
let itemsMap = {};
let monacoEditor = null;

$(document).ready(function() {
    Utils.initTheme();
    
    // Init Sort
    $('#sortSelect').val(currentSort);
    
    loadItems();

    // Search
    $('#searchInput').on('input', function() {
        loadItems();
    });

    // Sort
    $('#sortSelect').on('change', function() {
        currentSort = $(this).val();
        localStorage.setItem('copie_sort', currentSort);
        loadItems();
    });

    // Filters
    $('.filter-btn').on('click', function() {
        $('.filter-btn').removeClass('active btn-primary').addClass('btn-outline-secondary');
        $(this).removeClass('btn-outline-secondary').addClass('active btn-primary');
        currentFilter = $(this).data('type');
        loadItems();
    });

    // Initialize Monaco
    require.config({ paths: { 'vs': 'plugins/monaco-editor/min/vs' } });
    require(['vs/editor/editor.main'], function() {
        monacoEditor = monaco.editor.create(document.getElementById('monaco-container'), {
            value: '',
            language: 'plaintext',
            theme: $('html').attr('data-bs-theme') === 'dark' ? 'vs-dark' : 'vs',
            automaticLayout: true,
            minimap: { enabled: false },
            fontSize: 14,
            fontFamily: 'Monaco, Consolas, "Courier New", monospace',
            scrollBeyondLastLine: false
        });

        // Initialize Plugins
        if (typeof CopiePlugins !== 'undefined') {
            CopiePlugins.init(monacoEditor);
        }

        // Add shortcuts if needed
        monacoEditor.addCommand(monaco.KeyMod.CtrlCmd | monaco.KeyCode.KeyS, function() {
            saveItem();
        });

        // Raccourci Ctrl+Shift+S : Exporter
        monacoEditor.addCommand(monaco.KeyMod.CtrlCmd | monaco.KeyMod.Shift | monaco.KeyCode.KeyS, function() {
            exportCurrentItem();
        });

        // Raccourci Ctrl+Shift+D : Dupliquer (Ligne ou Sélection)
        monacoEditor.addCommand(monaco.KeyMod.CtrlCmd | monaco.KeyMod.Shift | monaco.KeyCode.KeyD, function() {
            var selection = monacoEditor.getSelection();
            var model = monacoEditor.getModel();
            
            if (selection.isEmpty()) {
                // Dupliquer la ligne courante
                var lineNumber = selection.startLineNumber;
                var lineContent = model.getLineContent(lineNumber);
                var lineMaxCol = model.getLineMaxColumn(lineNumber);
                
                // Insérer après la ligne courante
                var range = new monaco.Range(lineNumber, lineMaxCol, lineNumber, lineMaxCol);
                var text = '\n' + lineContent;
                
                monacoEditor.executeEdits("duplicate", [{
                    range: range,
                    text: text,
                    forceMoveMarkers: true
                }]);
            } else {
                // Dupliquer la sélection
                var text = model.getValueInRange(selection);
                var range = new monaco.Range(selection.endLineNumber, selection.endColumn, selection.endLineNumber, selection.endColumn);
                
                monacoEditor.executeEdits("duplicate", [{
                    range: range,
                    text: text,
                    forceMoveMarkers: true
                }]);
            }
        });

        // Raccourci Ctrl+Shift+Haut : Déplacer la ligne vers le haut
        monacoEditor.addCommand(monaco.KeyMod.CtrlCmd | monaco.KeyMod.Shift | monaco.KeyCode.UpArrow, function() {
            monacoEditor.trigger('keyboard', 'editor.action.moveLinesUpAction');
        });

        // Raccourci Ctrl+Shift+Bas : Déplacer la ligne vers le bas
        monacoEditor.addCommand(monaco.KeyMod.CtrlCmd | monaco.KeyMod.Shift | monaco.KeyCode.DownArrow, function() {
            monacoEditor.trigger('keyboard', 'editor.action.moveLinesDownAction');
        });

        // Raccourci Ctrl+Shift+L : Diviser la sélection en lignes (Multi-curseur)
        monacoEditor.addCommand(monaco.KeyMod.CtrlCmd | monaco.KeyMod.Shift | monaco.KeyCode.KeyL, function() {
            monacoEditor.trigger('keyboard', 'editor.action.insertCursorAtEndOfEachLineSelected', {});
        });

        // Raccourci Ctrl+K, Ctrl+U : Majuscules
        monacoEditor.addCommand(
            monaco.KeyMod.chord(monaco.KeyMod.CtrlCmd | monaco.KeyCode.KeyK, monaco.KeyMod.CtrlCmd | monaco.KeyCode.KeyU),
            function() { monacoEditor.getAction('editor.action.transformToUppercase').run(); }
        );

        // Raccourci Ctrl+K, Ctrl+L : Minuscules
        monacoEditor.addCommand(
            monaco.KeyMod.chord(monaco.KeyMod.CtrlCmd | monaco.KeyCode.KeyK, monaco.KeyMod.CtrlCmd | monaco.KeyCode.KeyL),
            function() { monacoEditor.getAction('editor.action.transformToLowercase').run(); }
        );

        // Raccourci Ctrl+Space : Ouvrir le sélecteur de snippets
        monacoEditor.addCommand(monaco.KeyMod.CtrlCmd | monaco.KeyCode.Space, function() {
            openSnippetSelector();
        });

        // Raccourci Ctrl+Shift+R : Aller au symbole
        monacoEditor.addCommand(monaco.KeyMod.CtrlCmd | monaco.KeyMod.Shift | monaco.KeyCode.KeyR, function() {
            openSymbolSelector();
        });

        // Register completion providers for all languages
        monaco.languages.getLanguages().forEach(function(lang) {
            monaco.languages.registerCompletionItemProvider(lang.id, {
                provideCompletionItems: function(model, position) {
                    const snippets = DB.getSnippets();
                    const currentLang = model.getLanguageId();
                    
                    // Filter snippets for the current language
                    const matchingSnippets = snippets.filter(s => s.language === currentLang);
                    
                    const suggestions = matchingSnippets.map(s => ({
                        label: s.label,
                        kind: monaco.languages.CompletionItemKind.Snippet,
                        documentation: s.description || '',
                        insertText: s.code,
                        insertTextRules: monaco.languages.CompletionItemInsertTextRule.InsertAsSnippet
                    }));

                    return { suggestions: suggestions };
                }
            });
        });

        // Check URL Hash for deep linking
        const hash = window.location.hash;
        if (hash === '#new') {
            openModal();
        } else if (hash.startsWith('#edit=')) {
            const id = hash.replace('#edit=', '');
            editItem(id);
        }
    });

    // Clear hash on modal close
    $('#editorModal').on('hidden.bs.modal', function () {
        history.pushState("", document.title, window.location.pathname + window.location.search);
    });

    // Snippet Search in Modal
    $('#snippetSearchInput').on('input', function() {
        renderSnippetList($(this).val());
    });

    // Keyboard navigation for Snippet Modal
    $('#snippetSearchInput').on('keydown', function(e) {
        const list = $('#snippetList');
        const items = list.find('button');
        let active = list.find('button.active');
        let index = items.index(active);
        
        if (e.key === 'ArrowDown') {
            e.preventDefault();
            if (index < items.length - 1) {
                active.removeClass('active');
                items.eq(index + 1).addClass('active')[0].scrollIntoView({ block: 'nearest' });
            } else if (index === -1 && items.length > 0) {
                 items.eq(0).addClass('active')[0].scrollIntoView({ block: 'nearest' });
            }
        } else if (e.key === 'ArrowUp') {
            e.preventDefault();
            if (index > 0) {
                active.removeClass('active');
                items.eq(index - 1).addClass('active')[0].scrollIntoView({ block: 'nearest' });
            }
        } else if (e.key === 'Enter') {
            e.preventDefault();
            if (active.length) {
                active.click();
            } else if (items.length > 0) {
                items.eq(0).click();
            }
        }
    });

    // Symbol Search in Modal
    $('#symbolSearchInput').on('input', function() {
        renderSymbolList($(this).val());
    });

    // Keyboard navigation for Symbol Modal
    $('#symbolSearchInput').on('keydown', function(e) {
        const list = $('#symbolList');
        const items = list.find('button');
        let active = list.find('button.active');
        let index = items.index(active);
        
        if (e.key === 'ArrowDown') {
            e.preventDefault();
            if (index < items.length - 1) {
                active.removeClass('active');
                items.eq(index + 1).addClass('active')[0].scrollIntoView({ block: 'nearest' });
            } else if (index === -1 && items.length > 0) {
                 items.eq(0).addClass('active')[0].scrollIntoView({ block: 'nearest' });
            }
        } else if (e.key === 'ArrowUp') {
            e.preventDefault();
            if (index > 0) {
                active.removeClass('active');
                items.eq(index - 1).addClass('active')[0].scrollIntoView({ block: 'nearest' });
            }
        } else if (e.key === 'Enter') {
            e.preventDefault();
            if (active.length) {
                active.click();
            } else if (items.length > 0) {
                items.eq(0).click();
            }
        }
    });

    $('#snippetSelectorModal').on('shown.bs.modal', function () {
        $('#snippetSearchInput').trigger('focus');
    });

    // Type change listener in Modal
    $('#itemType').on('change', function() {
        if (monacoEditor) {
            const type = $(this).val();
            const lang = Utils.getMonacoLanguage(type);
            monaco.editor.setModelLanguage(monacoEditor.getModel(), lang);
            if (typeof CopiePlugins !== 'undefined') CopiePlugins.updateVisibility(type);
        }
    });
});

function loadItems() {
    const search = $('#searchInput').val().toLowerCase();
    let items = DB.getCopies();

    // Filter
    if (currentFilter !== 'all') {
        items = items.filter(i => i.type === currentFilter);
    }

    // Search
    if (search) {
        items = items.filter(i => 
            i.title.toLowerCase().includes(search) || 
            (i.content && i.content.toLowerCase().includes(search))
        );
    }

    // Sort
    items.sort((a, b) => {
        if (currentSort === 'date_desc') {
            return new Date(b.updated_at || b.created_at) - new Date(a.updated_at || a.created_at);
        } else if (currentSort === 'date_asc') {
            return new Date(a.updated_at || a.created_at) - new Date(b.updated_at || b.created_at);
        } else if (currentSort === 'alpha_asc') {
            return a.title.localeCompare(b.title);
        } else if (currentSort === 'alpha_desc') {
            return b.title.localeCompare(a.title);
        }
        return 0;
    });

    renderGrid(items);
}

function renderGrid(items) {
    const grid = $('#itemsGrid');
    grid.empty();
    itemsMap = {};

    if (items.length === 0) {
        $('#emptyState').removeClass('d-none');
        return;
    }

    $('#emptyState').addClass('d-none');

    items.forEach(item => {
        itemsMap[item.id] = item;
        const card = `
            <div class="col-12">
                <div class="card item-card shadow-sm mb-3" ondblclick="editItem(${item.id})">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-start">
                            <div class="flex-grow-1">
                                <h5 class="card-title mb-1">
                                    <i class="bi ${getIconForType(item.type)} me-2"></i>
                                    ${Utils.escapeHtml(item.title)}
                                    <span class="badge bg-light text-dark border ms-2">
                                        ${item.type}
                                    </span>
                                </h5>
                                <div class="text-muted small">${Utils.formatDate(item.updated_at || item.created_at)}</div>
                            </div>
                            <div class="btn-group ms-2" role="group">
                                <button class="btn btn-sm btn-outline-secondary" onclick="toggleText(${item.id})" title="Afficher/Masquer">
                                    <i class="bi bi-chevron-down" id="icon-${item.id}"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-primary" onclick="copyToClipboard(this, ${item.id})" title="Copier">
                                    <i class="bi bi-clipboard"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-warning" onclick="editItem(${item.id})" title="Modifier">
                                    <i class="bi bi-pencil"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-danger" onclick="deleteItem(${item.id})" title="Supprimer">
                                    <i class="bi bi-trash"></i>
                                </button>
                            </div>
                        </div>
                        <div class="item-text mt-3 d-none" id="text-${item.id}">
                            <pre><code class="language-${item.type}">${Utils.escapeHtml(item.content)}</code></pre>
                        </div>
                    </div>
                </div>
            </div>
        `;
        grid.append(card);
    });
}

function getIconForType(type) {
    switch(type) {
        case 'php': return 'bi-filetype-php';
        case 'js': return 'bi-filetype-js';
        case 'html': return 'bi-filetype-html';
        case 'css': return 'bi-filetype-css';
        case 'python': return 'bi-filetype-py';
        case 'shell': return 'bi-terminal';
        case 'sql': return 'bi-database';
        case 'perl': return 'bi-file-code';
        case 'json': return 'bi-filetype-json';
        default: return 'bi-file-text';
    }
}

function toggleText(id) {
    const el = document.getElementById('text-' + id);
    const icon = document.getElementById('icon-' + id);
    if (el.classList.contains('d-none')) {
        el.classList.remove('d-none');
        icon.classList.remove('bi-chevron-down');
        icon.classList.add('bi-chevron-up');
        
        const codeBlock = el.querySelector('code');
        if (!codeBlock.dataset.highlighted && window.hljs) {
            hljs.highlightElement(codeBlock);
            codeBlock.dataset.highlighted = "yes";
        }
    } else {
        el.classList.add('d-none');
        icon.classList.remove('bi-chevron-up');
        icon.classList.add('bi-chevron-down');
    }
}

function saveQuickItem() {
    const title = $('#quickTitle').val();
    const type = $('#quickType').val();
    const content = $('#quickContent').val();

    if (!title || !content) return;

    const newItem = {
        title: title,
        type: type === 'auto' ? 'text' : type, // Simple auto fallback
        content: content
    };

    DB.saveCopy(newItem);
    $('#quickAddForm')[0].reset();
    Utils.showToast('Élément ajouté !');
    loadItems();
}

function openModal() {
    window.location.hash = 'new';
    $('#itemForm')[0].reset();
    $('#itemId').val(0);
    $('#btnDelete').addClass('d-none');
    
    if (monacoEditor) {
        monacoEditor.setValue('');
        monacoEditor.setScrollTop(0);
        monaco.editor.setModelLanguage(monacoEditor.getModel(), 'plaintext');
        if (typeof CopiePlugins !== 'undefined') CopiePlugins.updateVisibility('text');
    }
    
    $('#editorModal').modal('show');
}

function editItem(id) {
    const item = itemsMap[id];
    if (!item) return;

    window.location.hash = 'edit=' + id;
    $('#itemId').val(item.id);
    $('#itemTitle').val(item.title);
    $('#itemType').val(item.type);
    
    if (monacoEditor) {
        monacoEditor.setValue(item.content);
        monacoEditor.setScrollTop(0);
        const monacoLang = Utils.getMonacoLanguage(item.type);
        monaco.editor.setModelLanguage(monacoEditor.getModel(), monacoLang);
        if (typeof CopiePlugins !== 'undefined') CopiePlugins.updateVisibility(item.type);
    }
    
    $('#btnDelete').removeClass('d-none');
    $('#editorModal').modal('show');
}

function saveItem() {
    const form = $('#itemForm');
    if (!form[0].checkValidity()) {
        form[0].reportValidity();
        return;
    }

    const content = monacoEditor ? monacoEditor.getValue() : '';
    
    const data = {
        id: $('#itemId').val(),
        title: $('#itemTitle').val(),
        type: $('#itemType').val(),
        content: content
    };
    
    DB.saveCopy(data);
    // $('#editorModal').modal('hide'); // Keep open
    Utils.showToast('Enregistré !');
    loadItems();
}

async function exportCurrentItem() {
    const title = $('#itemTitle').val() || 'untitled';
    const type = $('#itemType').val();
    const content = monacoEditor ? monacoEditor.getValue() : '';
    
    const safeTitle = title.replace(/\s+/g, '_').replace(/[^a-zA-Z0-9_\-]/g, '');
    const extension = getExtensionForType(type);
    const filename = `${safeTitle}.${extension}`;

    // Try File System Access API
    if (window.showSaveFilePicker) {
        try {
            const handle = await window.showSaveFilePicker({
                suggestedName: filename,
                types: [{
                    description: type.toUpperCase() + ' File',
                    accept: { 'text/plain': ['.' + extension] },
                }],
            });
            const writable = await handle.createWritable();
            await writable.write(content);
            await writable.close();
            Utils.showToast('Fichier exporté !');
            return;
        } catch (err) {
            if (err.name !== 'AbortError') {
                console.error(err);
            } else {
                return; // User cancelled
            }
        }
    }

    // Fallback
    const blob = new Blob([content], { type: 'text/plain;charset=utf-8' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
}

function getExtensionForType(type) {
    const map = {
        'js': 'js',
        'text': 'txt',
        'markdown': 'md',
        'shell': 'sh',
        'python': 'py',
        'php': 'php',
        'html': 'html',
        'css': 'css',
        'sql': 'sql',
        'perl': 'pl',
        'json': 'json'
    };
    return map[type] || 'txt';
}

function getTypeFromExtension(ext) {
    const map = {
        'js': 'js',
        'txt': 'text',
        'md': 'markdown',
        'sh': 'shell',
        'bash': 'shell',
        'py': 'python',
        'pl': 'perl',
        'php': 'php',
        'html': 'html',
        'htm': 'html',
        'css': 'css',
        'json': 'json',
        'sql': 'sql'
    };
    return map[ext] || 'text';
}

function triggerImportFileIntoEditor() {
    const input = document.createElement('input');
    input.type = 'file';
    input.onchange = e => {
        const file = e.target.files[0];
        if (!file) return;

        const reader = new FileReader();
        reader.onload = function(e) {
            const content = e.target.result;
            
            // Open Modal as New
            openModal();

            if (monacoEditor) {
                monacoEditor.setValue(content);
            }
            
            // Update Title
            const filename = file.name;
            const lastDotIndex = filename.lastIndexOf('.');
            const name = lastDotIndex !== -1 ? filename.substring(0, lastDotIndex) : filename;
            const ext = lastDotIndex !== -1 ? filename.substring(lastDotIndex + 1).toLowerCase() : '';
            
            $('#itemTitle').val(name);
            
            // Try to detect type
            const type = getTypeFromExtension(ext);
            if (type) {
                $('#itemType').val(type).trigger('change');
            }
            
            Utils.showToast('Fichier importé !');
        };
        reader.readAsText(file);
    };
    input.click();
}

function deleteItem(id) {
    Utils.confirm('Voulez-vous vraiment supprimer cet élément ?', () => {
        DB.deleteCopy(id);
        Utils.showToast('Supprimé !');
        loadItems();
        $('#editorModal').modal('hide');
    });
}

function confirmDelete() {
    deleteItem($('#itemId').val());
}

function copyToClipboard(btn, id) {
    const item = itemsMap[id];
    if (!item) return;
    
    navigator.clipboard.writeText(item.content).then(() => {
        const icon = $(btn).find('i');
        icon.removeClass('bi-clipboard').addClass('bi-check-lg');
        Utils.showToast('Copié !');
        
        setTimeout(() => {
            icon.removeClass('bi-check-lg').addClass('bi-clipboard');
        }, 2000);
    });
}

function transformText(type) {
    if (!monacoEditor) return;
    if (type === 'upper') {
        monacoEditor.getAction('editor.action.transformToUppercase').run();
    } else if (type === 'lower') {
        monacoEditor.getAction('editor.action.transformToLowercase').run();
    }
}

function toggleWordWrap() {
    if (!monacoEditor) return;
    const current = monacoEditor.getOption(monaco.editor.EditorOption.wordWrap);
    const newState = current === 'off' ? 'on' : 'off';
    monacoEditor.updateOptions({ wordWrap: newState });
}

// Snippet Selector Logic
let currentSnippetContextLang = null;

function openSnippetSelector() {
    $('#snippetSelectorModal').modal('show');
    
    currentSnippetContextLang = null;
    if (monacoEditor) {
        const lang = monacoEditor.getModel().getLanguageId();
        if (lang && lang !== 'plaintext') {
            currentSnippetContextLang = lang;
        }
    }
    
    $('#snippetSearchInput').val('');
    renderSnippetList('');
}

function renderSnippetList(search = '') {
    const snippets = DB.getSnippets();
    const list = $('#snippetList');
    list.empty();

    let filtered = snippets;

    // Context Filter (Language)
    if (currentSnippetContextLang) {
        filtered = filtered.filter(s => s.language === currentSnippetContextLang);
    }

    // Search Filter
    if (search) {
        const term = search.toLowerCase();
        filtered = filtered.filter(s => 
            s.label.toLowerCase().includes(term) || 
            (s.description && s.description.toLowerCase().includes(term))
        );
    }

    if (filtered.length === 0) {
        $('#noSnippetsMsg').removeClass('d-none');
        return;
    }
    $('#noSnippetsMsg').addClass('d-none');

    filtered.forEach((s, index) => {
        const activeClass = index === 0 ? 'active' : '';
        const item = `
            <button type="button" class="list-group-item list-group-item-action ${activeClass}" onclick="insertSnippet(${s.id})">
                <div class="d-flex w-100 justify-content-between">
                    <h6 class="mb-1 fw-bold font-monospace">${Utils.escapeHtml(s.label)}</h6>
                    <small class="text-muted">${s.language}</small>
                </div>
                <p class="mb-1 small text-muted">${Utils.escapeHtml(s.description || '')}</p>
            </button>
        `;
        list.append(item);
    });
}

function insertSnippet(id) {
    const snippet = DB.getSnippet(id);
    if (snippet && monacoEditor) {
        const contribution = monacoEditor.getContribution('snippetController2');
        if (contribution) {
            contribution.insert(snippet.code);
        }
        $('#snippetSelectorModal').modal('hide');
    }
}

// Symbol Selector Logic
let currentSymbols = [];

function openSymbolSelector() {
    if (!monacoEditor) return;
    
    currentSymbols = CopiePlugins.getSymbols();
    if (currentSymbols.length === 0) {
        Utils.showToast('Aucun symbole trouvé dans ce fichier', 'warning');
        return;
    }

    $('#goToSymbolModal').modal('show');
    $('#symbolSearchInput').val('');
    renderSymbolList('');
    
    // Focus input after modal shown
    setTimeout(() => $('#symbolSearchInput').focus(), 500);
}

function renderSymbolList(search = '') {
    const list = $('#symbolList');
    list.empty();

    let filtered = currentSymbols;

    if (search) {
        const term = search.toLowerCase();
        filtered = filtered.filter(s => s.text.toLowerCase().includes(term));
    }

    if (filtered.length === 0) {
        $('#noSymbolsMsg').removeClass('d-none');
        return;
    }
    $('#noSymbolsMsg').addClass('d-none');

    filtered.forEach((s, index) => {
        const activeClass = index === 0 ? 'active' : '';
        const item = `
            <button type="button" class="list-group-item list-group-item-action ${activeClass}" onclick="goToSymbol(${s.line})">
                <div class="d-flex w-100 justify-content-between align-items-center">
                    <div>
                        <span class="badge bg-secondary me-2">${s.type}</span>
                        <span class="fw-bold font-monospace">${Utils.escapeHtml(s.text)}</span>
                    </div>
                    <small class="text-muted">Ligne ${s.line}</small>
                </div>
                <small class="text-muted d-block text-truncate" style="max-width: 90%;">${Utils.escapeHtml(s.fullLine)}</small>
            </button>
        `;
        list.append(item);
    });
}

function goToSymbol(line) {
    if (monacoEditor) {
        monacoEditor.revealLineInCenter(line);
        monacoEditor.setPosition({ lineNumber: line, column: 1 });
        monacoEditor.focus();
        $('#goToSymbolModal').modal('hide');
    }
}

// Export/Import UI Handlers
function triggerExport() {
    DB.exportData();
}

function triggerImport() {
    const input = document.createElement('input');
    input.type = 'file';
    input.accept = '.txt,.json';
    input.onchange = e => {
        const file = e.target.files[0];
        if (!file) return;
        const reader = new FileReader();
        reader.onload = event => {
            if (DB.importData(event.target.result)) {
                Utils.showToast('Import réussi !');
                loadItems();
            } else {
                Utils.showToast('Erreur lors de l\'import', 'danger');
            }
        };
        reader.readAsText(file);
    };
    input.click();
}
