// State
let diffEditor = null;
let originalModel = null;
let modifiedModel = null;

// Init
$(document).ready(function() {
    // Theme Init
    Utils.initTheme();

    // Init Monaco
    require.config({ paths: { 'vs': 'plugins/monaco-editor/min/vs' }});
    require(['vs/editor/editor.main'], function() {
        
        // Create Diff Editor
        diffEditor = monaco.editor.createDiffEditor(document.getElementById('diff-editor-container'), {
            originalEditable: true, // Allow editing left side too if needed
            automaticLayout: true,
            scrollBeyondLastLine: false,
            theme: $('html').attr('data-bs-theme') === 'dark' ? 'vs-dark' : 'vs'
        });

        // Create Models
        originalModel = monaco.editor.createModel("", "text/plain");
        modifiedModel = monaco.editor.createModel("", "text/plain");

        diffEditor.setModel({
            original: originalModel,
            modified: modifiedModel
        });

        // Add shortcuts to both editors
        [diffEditor.getOriginalEditor(), diffEditor.getModifiedEditor()].forEach(editor => {
            editor.addCommand(monaco.KeyMod.CtrlCmd | monaco.KeyMod.Shift | monaco.KeyCode.UpArrow, function() {
                editor.trigger('keyboard', 'editor.action.moveLinesUpAction');
            });
            editor.addCommand(monaco.KeyMod.CtrlCmd | monaco.KeyMod.Shift | monaco.KeyCode.DownArrow, function() {
                editor.trigger('keyboard', 'editor.action.moveLinesDownAction');
            });
        });

        // Listen for theme changes to update editor
        const observer = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
                if (mutation.attributeName === "data-bs-theme") {
                    const theme = $('html').attr('data-bs-theme');
                    monaco.editor.setTheme(theme === 'dark' ? 'vs-dark' : 'vs');
                }
            });
        });
        observer.observe(document.documentElement, { attributes: true });
    });
});

// File Loading
function loadFile(side) {
    const input = document.getElementById(side === 'left' ? 'fileLeft' : 'fileRight');
    const file = input.files[0];
    
    if (!file) return;

    const reader = new FileReader();
    reader.onload = function(e) {
        const text = e.target.result;
        
        if (side === 'left') {
            if (originalModel) originalModel.setValue(text);
        } else {
            if (modifiedModel) modifiedModel.setValue(text);
        }
    };
    reader.readAsText(file);
}

// Export
function exportModified() {
    if (!modifiedModel) return;
    
    const content = modifiedModel.getValue();
    if (!content) {
        Utils.showToast("Le fichier est vide", "warning");
        return;
    }

    const blob = new Blob([content], {type: "text/plain;charset=utf-8"});
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    
    // Try to get filename from input if possible, else default
    const fileInput = document.getElementById('fileRight');
    let filename = "modified_file.txt";
    if (fileInput.files.length > 0) {
        filename = "fixed_" + fileInput.files[0].name;
    }
    
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
    
    Utils.showToast("Fichier exporté avec succès");
}
