/**
 * Shared Logic for Local Copie
 * Handles Data Persistence (localStorage), Theme, and Utilities.
 */

const DB_KEY = 'copie_db';

const DB = {
    data: {
        copies: [],
        snippets: []
    },

    init: function() {
        const stored = localStorage.getItem(DB_KEY);
        if (stored) {
            try {
                this.data = JSON.parse(stored);
                // Ensure structure
                if (!this.data.copies) this.data.copies = [];
                if (!this.data.snippets) this.data.snippets = [];
            } catch (e) {
                console.error("Error parsing localStorage", e);
                this.loadFromInitial();
            }
        } else {
            this.loadFromInitial();
        }
    },

    loadFromInitial: function() {
        if (window.DATA_DB) {
            this.data = JSON.parse(JSON.stringify(window.DATA_DB)); // Deep copy
            this.save();
        }
    },

    save: function() {
        localStorage.setItem(DB_KEY, JSON.stringify(this.data));
    },

    // Copies CRUD
    getCopies: function() {
        return this.data.copies.sort((a, b) => new Date(b.updated_at || b.created_at) - new Date(a.updated_at || a.created_at));
    },

    getCopy: function(id) {
        return this.data.copies.find(c => c.id == id);
    },

    saveCopy: function(copy) {
        if (!copy.id || copy.id == 0) {
            copy.id = Date.now(); // Simple ID generation
            copy.created_at = new Date().toISOString();
        }
        copy.updated_at = new Date().toISOString();
        
        const index = this.data.copies.findIndex(c => c.id == copy.id);
        if (index > -1) {
            this.data.copies[index] = { ...this.data.copies[index], ...copy };
        } else {
            this.data.copies.push(copy);
        }
        this.save();
        return copy;
    },

    deleteCopy: function(id) {
        this.data.copies = this.data.copies.filter(c => c.id != id);
        this.save();
    },

    // Snippets CRUD
    getSnippets: function() {
        return this.data.snippets;
    },

    getSnippet: function(id) {
        return this.data.snippets.find(s => s.id == id);
    },

    saveSnippet: function(snippet) {
        if (!snippet.id || snippet.id == 0) {
            snippet.id = Date.now();
        }
        
        const index = this.data.snippets.findIndex(s => s.id == snippet.id);
        if (index > -1) {
            this.data.snippets[index] = { ...this.data.snippets[index], ...snippet };
        } else {
            this.data.snippets.push(snippet);
        }
        this.save();
        return snippet;
    },

    deleteSnippet: function(id) {
        this.data.snippets = this.data.snippets.filter(s => s.id != id);
        this.save();
    },

    // Export / Import
    exportData: function() {
        // Include current theme in export
        if (!this.data.config) this.data.config = {};
        this.data.config.theme = localStorage.getItem('theme') || 'light';

        const jsonContent = JSON.stringify(this.data, null, 4);
        const jsContent = `window.DATA_DB = ${jsonContent};`;
        
        const dataStr = "data:text/plain;charset=utf-8," + encodeURIComponent(jsContent);
        const downloadAnchorNode = document.createElement('a');
        downloadAnchorNode.setAttribute("href", dataStr);
        downloadAnchorNode.setAttribute("download", "data.txt");
        document.body.appendChild(downloadAnchorNode);
        downloadAnchorNode.click();
        downloadAnchorNode.remove();
    },

    importData: function(content) {
        try {
            // Try parsing as JSON first (legacy backup)
            let parsed;
            try {
                parsed = JSON.parse(content);
            } catch (e) {
                // Try parsing as JS file (window.DATA_DB = ...)
                const match = content.match(/window\.DATA_DB\s*=\s*({[\s\S]*});?/);
                if (match && match[1]) {
                    parsed = JSON.parse(match[1]);
                }
            }

            if (parsed && parsed.copies && parsed.snippets) {
                this.data = parsed;
                this.save();
                
                // Apply imported theme if exists
                if (this.data.config && this.data.config.theme) {
                    Utils.setTheme(this.data.config.theme);
                }

                return true;
            }
            return false;
        } catch (e) {
            console.error("Import error", e);
            return false;
        }
    }
};

// Initialize DB immediately
DB.init();


// Shared UI Utils
const Utils = {
    showToast: function(message, type = 'success') {
        const toastEl = document.getElementById('liveToast');
        if (!toastEl) return;
        
        const toast = new bootstrap.Toast(toastEl);
        const msgEl = document.getElementById('toastMsg');
        if (msgEl) msgEl.textContent = message;
        
        toastEl.className = `toast align-items-center text-white bg-${type} border-0`;
        toast.show();
    },

    confirm: function(message, callback) {
        const modalEl = document.getElementById('confirmModal');
        const msgEl = document.getElementById('confirmMessage');
        const btnConfirm = document.getElementById('btnConfirmAction');
        
        if (!modalEl || !msgEl || !btnConfirm) {
            if (confirm(message)) callback();
            return;
        }

        msgEl.textContent = message;
        const modal = new bootstrap.Modal(modalEl);
        
        // Clean previous events
        const newBtn = btnConfirm.cloneNode(true);
        btnConfirm.parentNode.replaceChild(newBtn, btnConfirm);
        
        newBtn.addEventListener('click', () => {
            callback();
            modal.hide();
        });
        
        modal.show();
    },

    formatDate: function(dateString) {
        if (!dateString) return '';
        const date = new Date(dateString);
        if (isNaN(date.getTime())) return dateString;
        return new Intl.DateTimeFormat('fr-FR', { 
            day: '2-digit', month: 'short', hour: '2-digit', minute: '2-digit' 
        }).format(date);
    },

    escapeHtml: function(text) {
        if (text == null) return '';
        return String(text)
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    },

    initTheme: function() {
        let defaultTheme = 'light';
        if (window.DATA_DB && window.DATA_DB.config && window.DATA_DB.config.theme) {
            defaultTheme = window.DATA_DB.config.theme;
        }
        
        const savedTheme = localStorage.getItem('theme') || defaultTheme;
        this.setTheme(savedTheme);
        
        $('#themeToggle').on('click', () => {
            const current = $('html').attr('data-bs-theme');
            const next = current === 'dark' ? 'light' : 'dark';
            this.setTheme(next);
        });
    },

    setTheme: function(theme) {
        $('html').attr('data-bs-theme', theme);
        localStorage.setItem('theme', theme);
        
        const icon = $('#themeToggle i');
        const hljsLink = document.getElementById('hljs-theme');

        if (theme === 'dark') {
            icon.removeClass('bi-moon-fill').addClass('bi-sun-fill');
            if (window.monacoEditor) window.monaco.editor.setTheme('vs-dark');
            if (hljsLink) hljsLink.href = 'plugins/hightlight/styles/github-dark.css';
        } else {
            icon.removeClass('bi-sun-fill').addClass('bi-moon-fill');
            if (window.monacoEditor) window.monaco.editor.setTheme('vs');
            if (hljsLink) hljsLink.href = 'plugins/hightlight/styles/github.css';
        }
    },

    getMonacoLanguage: function(type) {
        const map = {
            'js': 'javascript',
            'text': 'plaintext',
            'markdown': 'markdown',
            'shell': 'shell',
            'python': 'python',
            'php': 'php',
            'html': 'html',
            'css': 'css',
            'sql': 'sql',
            'perl': 'perl',
            'json': 'json'
        };
        return map[type] || 'plaintext';
    }
};
